grammar EOSEx ;

/*
 * Parser Rules
 */calculationScript
	: (expression | variableDefinition* returnResult) EOF ;

function
	: (FUNCTION | UW) '(' functionArguments functionParameters? ')' ;

ifThenElse
	: IF '(' expression ')' (expression | ('{' expression '}')) ELSE (expression | ('{' expression '}')) ;

functionArguments
	: expression?
	| (expression (  ','  expression )*) ;

functionParameters
	: '@' functionParameter (',' functionParameter)* ;

functionParameter
	: functionParameterName ( '=' functionParameterValue )? ;

functionParameterName
	: UW ;

functionParameterValue
	: UW
	| NUMBER
	| CONSTANT
	| NAN ;

propertyRef
	: REF_PREFIX PROPERTY '->' (SEARCH_POLICY | LW) ;

variableRef
	: REF_PREFIX VARIABLE '->' (SEARCH_POLICY | LW) ;

characteristic2DRef
	: REF_PREFIX CHARACTERISTIC_2D '->' (SEARCH_POLICY | LW) '(' expression ')' ;
	
characteristic3DRef
	: REF_PREFIX CHARACTERISTIC_3D ZW_SELECTION? '->' (SEARCH_POLICY | LW) '(' expression ',' expression ')' ;
	
characteristic3DExtremeLineRef
	: REF_PREFIX CHARACTERISTIC_3D EXTREME_LINE_XY_SELECTION '->' (SEARCH_POLICY | LW) '(' expression ')' ;

expression
	: LW													#variableNameExpr
	| NUMBER												#numberExpr
	| CONSTANT												#constantExpr
	| NAN													#nanExpr
	| TEXT													#textExpr
	| function												#functionExpr
	| propertyRef											#propertyRefExpr
	| variableRef											#variableRefExpr
	| characteristic2DRef									#characteristic2DRefExpr
	| characteristic3DRef									#characteristic3DRefExpr
	| characteristic3DExtremeLineRef						#characteristic3DExtremeLineRefExpr
	| ifThenElse											#ifThenElseExpr
	| '!' expression										#notExpr
	| expression operand=POWER expression					#powerExpr
	| expression operand=(MUL | DIV) expression				#mulDivExpr
	| expression operand=(ADD | SUB) expression				#addSubExpr
	| expression operand=(GTE | GT | LTE | LT) expression	#compareGtLtExpr
	| expression operand=(EQ | NEQ) expression				#compareEqNeqExpr
	| expression operand=(BITAND | BITOR) expression		#bitAndOrExpr
	| expression operand=(AND | OR) expression				#LogicalAndOrExpr
	| '(' expression ')'									#parensExpr ;

variableDefinition
	: VAR LW '=' expression ';' ;

returnResult
	: RETURN expression ';' ;

/*
 * Lexer Rules
 */

fragment LOWERCASE  : [a-z];
fragment ZW01  : [zwZW01];
fragment XY  : [xyXY];
fragment MINMAX  : 'Min' | 'min' | 'Max' | 'max';
fragment UPPERCASE  : [A-Z];
fragment DIGIT  : [0-9];
fragment LETTER : [a-zA-Z\u0080-\uFFFF];
fragment REF_PART:
	LETTER (LETTER | DIGIT | '_' | '/')*;
fragment ESC : '\\"'
	| '\\\\' ;

NUMBER
	: '-'? DIGIT+ ('.' DIGIT+)? 
	| '-'? DIGIT ('.' DIGIT+)? 'E' '-'? DIGIT+ ;

VAR
	: 'var' ;

RETURN
	: 'return' ;

IF
	: 'if' ;

ELSE
	: 'else' ;

CONSTANT
	: 'TRUE'
	| 'FALSE'
	| 'PI' 
	| 'RASTER' ;

NAN
	: 'NaN' ;

FUNCTION
	: 'Abs'
	| 'And'
	| 'Avg'
	| 'Average'
	| 'BitAnd'
	| 'BitOr'
	| 'Ceil'
	| 'Diff'
	| 'Div'
	| 'EndTimestamp'
	| 'Eq'
	| 'Floor'
	| 'Ge'
	| 'Gt'
	| 'If'
	| 'IsValid'
	| 'PrevValid'
	| 'LastValid'
	| 'NextValid'
	| 'LastValidValueInRaster'
	| 'Le'
	| 'Lt'
	| 'LocalHour'
	| 'LocalFractionalHour'
	| 'LocalWeekDay'
	| 'LocalDayOfMonth'
	| 'LocalDayOfYear'
	| 'NextValue'
	| 'PrevValue'
	| 'VirtualZoneHour'
	| 'VirtualZoneDayOfMonth'
	| 'VirtualZoneDayOfYear'
	| 'VirtualZoneFractionalDayOfYear'
	| 'VirtualZoneYear'
	| 'Map'
	| 'Mult'
	| 'Min'
	| 'Max'
	| 'Merge'
	| 'Median'
	| 'MovingAverage'
	| 'MovingMedian'
	| 'Not'
	| 'NotEq'
	| 'Negate'
	| 'Or'
	| 'Power'
	| 'Random'
	| 'ReBound'
	| 'TimeInterval'
	| 'LowerLimit'
	| 'UpperLimit'
	| 'Quality'
	| 'Sign'
	| 'Sum'
	| 'SumDiff'
	| 'Integral'
	| 'IntegralKum'
	| 'Rate'
	| 'SumKum'
	| 'Differential'
	| 'Delta'
	| 'WindowCounter'
	| 'Sin'
	| 'Cos'
	| 'Tan' ;

SEARCH_POLICY
	: 'Auto'
	| 'Parent'
	| 'Children'
	| 'ThisOrParent'
	| 'ThisOrChildren'
	| 'This'
	| 'DirectChildren'
	| 'TopChildren'
	| 'All' ;

/* LOWER_WORD */LW
	: LOWERCASE (LOWERCASE | UPPERCASE | DIGIT | '_')* ;

/* UPPER_WORD */UW
	: UPPERCASE (LOWERCASE | UPPERCASE | DIGIT | '_')* ;

REF_PREFIX
	: '$' ;

PROPERTY
	: REF_PART'.'REF_PART ;
	
VARIABLE
	: REF_PART'.'REF_PART'.'REF_PART'.'REF_PART ;

CHARACTERISTIC_2D
	: REF_PART'.'REF_PART'.'REF_PART'~'REF_PART ;

CHARACTERISTIC_3D
	: REF_PART '.' REF_PART '.<' REF_PART ',' REF_PART '>~<' (REF_PART (','REF_PART)?) '>' ('_'REF_PART)? ;

ZW_SELECTION
	: '[' ZW01 ']' ;
	
EXTREME_LINE_XY_SELECTION
	: '[' XY 'over' MINMAX ZW01 ']' ;

POWER
	: '^' ;

MUL
	: '*' ;

DIV
	: '/' ;

ADD
	: '+' ;

SUB
	: '-' ;

GTE
	: '>=' ;

GT
	: '>' ;

LTE
	: '<=' ;

LT
	: '<' ;

EQ
	: '==' ;

NEQ
	: '!=' ;

AND
	: '&&' ;

OR
	: '||' ;

BITAND
	: '&' ;

BITOR
	: '|' ;

WHITESPACE
	: (' ' | '\t')+ -> skip ;

NEWLINE
	: ('\r'? '\n' | '\r')+ -> skip ;

COMMENT
	: '/*' .*? '*/' -> skip ;

LINE_COMMENT
	: '//' ~[\r\n]* -> skip ;

TEXT
	: '"' ( ESC | ~[\\"\r\n] )* '"' ;

